<?php

// app/Http/Controllers/Dashboard/DeviceController.php
namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use App\Models\Device;
use Jenssegers\Agent\Agent;

class DeviceController extends Controller
{
    public function index(Request $request)
    {
        // The TrackUserDevice middleware handles device creation/updates.
        // We can just fetch the devices, ordering by the most recently active.
        $devices = Auth::user()->devices()
            ->orderByDesc('last_activity')->get();
        return view('dashboard.devices', compact('devices'));
    }

    public function revoke(Request $request, $id)
    {
        $device = Auth::user()->devices()->findOrFail($id);

        $device->delete();

        return back()->with('success', 'Device has been revoked successfully.');
    }

    public function addCurrentDevice(Request $request)
    {
        $request->validate([
            'device_name' => ['required', 'string', 'max:255'],
        ]);

        $user = Auth::user();

        $agent = new Agent();

        // Create the new device first
        $user->devices()->create([
            'name' => $request->input('device_name'),
            'session_id' => $request->session()->getId(), // Use current session ID for now
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'device_type' => $agent->isDesktop() ? 'Desktop' : ($agent->isTablet() ? 'Tablet' : 'Mobile'),
            'browser' => $agent->browser(),
            'platform' => $agent->platform(),
            'last_activity' => now(),
        ]);

        // Now, log the user out and back in to create a new session and associate it with the new device.
        // This will invalidate the old session.
        Auth::logout();
        Auth::login($user, true); // Log the user back in, "remembering" them.

        return redirect()->route('user.devices')->with('success', 'New device has been added successfully.');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'device_name' => ['required', 'string', 'max:255'],
        ]);

        $device = Auth::user()->devices()->findOrFail($id);
        $device->update(['name' => $request->input('device_name')]);

        return back()->with('success', 'Device name has been updated successfully.');
    }
}
