<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\WalletSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WalletSettingController extends Controller
{
    /**
     * Show the wallet settings form for a specific wallet type.
     */
    public function show(Request $request, string $wallet)
    {
        $user = Auth::user();
        $activeBrandId = $request->query('brand_id');

        // If no brand_id is in the query, and the user has brands, redirect to the first active brand.
        if (!$activeBrandId) {
            $firstBrand = Brand::where('user_id', $user->id)->where('status', 1)->first();
            if ($firstBrand) {
                return redirect()->route(auth()->user()->role === 'admin' ? 'admin.wallets.settings' : 'user.wallets.settings', ['wallet' => $wallet, 'brand_id' => $firstBrand->id]);
            }
        }
        if (!$activeBrandId && Brand::where('user_id', $user->id)->exists()) {
            abort(404, 'Please select a brand.');
        }

        // Find the wallet setting for the current user, wallet type, and brand.
        // If it doesn't exist, create a new instance without saving it yet.
        $walletSetting = $user->walletSettings()
            ->where('wallet_type', $wallet)
            ->where('brand_id', $activeBrandId)
            ->first();

        if (!$walletSetting) $walletSetting = new WalletSetting(['wallet_type' => $wallet, 'brand_id' => $activeBrandId, 'user_id' => $user->id, 'status' => false]);

        // Determine wallet categories for conditional rendering in the view
        $isMobileBanking = in_array($wallet, [
            'bkash', 'nagad', 'rocket', 'upay',
            'cellfin', 'tap', 'okwallet', 'surecash',
            'mcash', 'mycash', 'ipay'
        ]);
        $isBank = in_array($wallet, [
            'ibl', 'abbank', 'citybank', 'sonali',
            'dbbl', 'basic', 'ebl', 'bbrac',
            'basia', 'agrani', 'jamuna', 'ific'
        ]);
        $isCrypto = in_array($wallet, [
            'binance', 'payeer'
        ]);

        // Prepare wallet name for view
        $walletName = ucfirst($wallet);

        return view('dashboard.wallets', compact(
            'wallet',
            'walletName',
            'walletSetting',
            'isMobileBanking',
            'isBank',
            'isCrypto'
        ));
    }

    /**
     * Update the wallet settings for a specific wallet type.
     */
    public function update(Request $request, string $wallet)
    {
        $user = Auth::user();

        // Base validation rules
        $rules = [
            'status' => ['required', 'boolean'],
            'brand_id' => ['required', 'integer', 'exists:brands,id'],
        ];

        // Add specific rules based on wallet type
        if (in_array($wallet, [
            'bkash', 'nagad', 'rocket', 'upay',
            'cellfin', 'tap', 'okwallet', 'surecash',
            'mcash', 'mycash', 'ipay'
        ])) {
            $rules = array_merge($rules, [
                'is_personal_active' => ['boolean'],
                'is_agent_active'   => ['boolean'],
                'is_merchant_active'=> ['boolean'],
                'is_payment_active' => ['boolean'],
                'personal_number'   => ['nullable', 'string', 'max:255'],
                'payment_number'    => ['nullable', 'string', 'max:255'],
                'agent_number'      => ['nullable', 'string', 'max:255'],
                'sandbox'           => ['boolean'],
                'logs'              => ['boolean'],
                'username'          => ['nullable', 'string', 'max:255'],
                'password'          => ['nullable', 'string', 'max:255'],
                'app_key'           => ['nullable', 'string', 'max:255'],
                'app_secret'        => ['nullable', 'string', 'max:255'],
            ]);
        } elseif (in_array($wallet, [
            'ibl', 'abbank', 'citybank', 'sonali',
            'dbbl', 'basic', 'ebl', 'bbrac',
            'basia', 'agrani', 'jamuna', 'ific'
        ])) {
            $rules = array_merge($rules, [
                'account_number' => ['nullable', 'string', 'max:255'],
                'account_name'   => ['nullable', 'string', 'max:255'],
                'branch_name'    => ['nullable', 'string', 'max:255'],
            ]);
        } elseif (in_array($wallet, [
            'binance', 'payeer'
        ])) {
            $rules = array_merge($rules, [
                'wallet_address' => ['nullable', 'string', 'max:255'],
                'api_key'        => ['nullable', 'string', 'max:255'],
            ]);
        }

        $validatedData = $request->validate($rules);

        // If the password field was not included in the request (e.g., disabled input),
        // unset it from the validated data to avoid overwriting the existing one.
        // If it was submitted as an empty string, it will be saved as null, allowing users to clear it.
        if (!$request->has('password')) {
            unset($validatedData['password']);
        }
        if (!$request->has('app_secret')) {
            unset($validatedData['app_secret']);
        }

        // Find or create/update the wallet setting record
        $brandId = $validatedData['brand_id'];

        $walletSetting = $user->walletSettings()->updateOrCreate(
            ['wallet_type' => $wallet, 'brand_id' => $brandId],
            $validatedData
        );
        return response()->json([
            'message'       => ucfirst($wallet) . ' settings updated successfully!',
            'walletSetting' => $walletSetting,
        ]);
    }
}
