<?php

namespace Ihasan\Bkash;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Route;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class BkashServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        /*
         * This class is a Package Service Provider
         *
         * More info: https://github.com/spatie/laravel-package-tools
         */
        $package
            ->name('laravel-bkash')
            ->hasConfigFile()
            ->hasViews('bkash')
            ->hasMigrations([
                'create_bkash_payments_table',
                'create_bkash_refunds_table',
            ]);
    }

    public function packageRegistered(): void
    {
        $this->app->singleton(Bkash::class, function ($app) {
            return new Bkash;
        });
    }

    public function packageBooted(): void
    {
        Http::macro('bkash', function (?string $token = null, ?array $credentials = null) {
            $baseUrl = config('bkash.sandbox')
                ? config('bkash.sandbox_base_url')
                : config('bkash.live_base_url');

            $version = config('bkash.version');

            // Use provided credentials if available, otherwise fall back to config
            $currentCredentials = $credentials ?? config('bkash.credentials');

            $client = Http::baseUrl("{$baseUrl}")->withHeaders([
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ]);
    
            if ($token) {
                $client = $client->withHeaders([
                    'Authorization' => $token,
                    'X-APP-Key' => $currentCredentials['app_key'],
                ]);
            }

            return $client;
        });

        $this->registerRoutes();

        if ($this->app->runningInConsole()) {

            $this->commands([
                \Ihasan\Bkash\Commands\BkashSetupCommand::class,
            ]);


            $this->publishes([
                __DIR__.'/../resources/views' => resource_path('views/vendor/bkash'),
            ], 'bkash-views');

            $this->publishes([
                __DIR__.'/Http/Controllers' => app_path('Http/Controllers/Vendor/Bkash'),
            ], 'bkash-controllers');

            $this->publishes([
                __DIR__.'/../config/bkash.php' => config_path('bkash.php'),
            ], 'config');

            $this->publishes([
                __DIR__.'/../database/migrations/create_bkash_payments_table.php.stub' => $this->getMigrationFileName('create_bkash_payments_table.php'),
                __DIR__.'/../database/migrations/create_bkash_refunds_table.php.stub' => $this->getMigrationFileName('create_bkash_refunds_table.php'),
                __DIR__.'/../database/migrations/update_bkash_tables_with_prefix.php.stub' => $this->getMigrationFileName('update_bkash_tables_with_prefix.php'),
            ], 'migrations');

        }
    }

    protected function getMigrationFileName($migrationFileName): string
    {
        $timestamp = date('Y_m_d_His');

        return database_path('migrations/' . $timestamp . '_' . $migrationFileName);
    }

    protected function registerRoutes(): void
    {
        if (config('bkash.routes.enabled', true)) {
            Route::group($this->routeConfiguration(), function () {
                $this->loadRoutesFrom(__DIR__.'/routes/web.php');
            });
        }
    }

    protected function routeConfiguration(): array
    {
        return [
            'prefix' => config('bkash.routes.prefix', 'bkash'),
            'middleware' => config('bkash.routes.middleware', ['web']),
            'as' => 'bkash.',
        ];
    }
}
